package ch.frankel.blog.roo.vaadin.web;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import com.vaadin.annotations.AutoGenerated;
import com.vaadin.spring.roo.addon.annotations.RooVaadinEntityManagerView;
import com.vaadin.spring.roo.addon.annotations.RooVaadinEntityView;
import com.vaadin.ui.AbsoluteLayout;
import com.vaadin.ui.Alignment;
import com.vaadin.ui.Button;
import com.vaadin.ui.Button.ClickEvent;
import com.vaadin.ui.Component;
import com.vaadin.ui.ComponentContainer;
import com.vaadin.ui.CssLayout;
import com.vaadin.ui.CustomComponent;
import com.vaadin.ui.HorizontalSplitPanel;
import com.vaadin.ui.Label;
import com.vaadin.ui.NativeButton;
import com.vaadin.ui.Panel;
import com.vaadin.ui.VerticalLayout;
import com.vaadin.ui.themes.Reindeer;

/**
 * Entity manager view that by default automatically discovers all entity views
 * (classes with the {@link RooVaadinEntityView} annotation) and lets the user
 * choose one of them to display.
 * 
 * This class is designed to be compatible with the Vaadin Visual Editor.
 * 
 * All methods and fields not marked with the {@link AutoGenerated} annotation
 * can be modified freely.
 * 
 * If you are planning to use the visual editor, do not touch the methods marked
 * with the {@link AutoGenerated} annotation - they will be removed and
 * re-generated when saving the class from the visual editor. Instead, add any
 * custom code to methods called from the constructor after the initial view
 * construction.
 * 
 * If you will not use the Vaadin Visual Editor to edit this class, all the
 * {@link AutoGenerated} annotations can be removed.
 */
@RooVaadinEntityManagerView
public class WebEntityManagerView extends CustomComponent {

    @AutoGenerated
    private HorizontalSplitPanel mainLayout;
    @AutoGenerated
    private VerticalLayout initialView;
    @AutoGenerated
    private CssLayout viewList;

    private List<EntityMenuItem> menuItems = new ArrayList<EntityMenuItem>();

    private Component currentView;

    private class EntityMenuItem extends AbsoluteLayout {
        private Button selectButton = new NativeButton();
        private Button newButton = new NativeButton("New");
        private Object view;

        public EntityMenuItem(String entityName, Object view) {
            setHeight("36px");
            setWidth("100%");
            addStyleName("menu-item");
            selectButton.setCaption(entityName);
            this.view = view;
            newButton.addStyleName("new");
            newButton.setVisible(false);
            newButton.setEnabled(false);
            addComponent(selectButton, "top: 0; left: 0; right: 0;");
            addComponent(newButton, "top: 8px; right: 9px;");

            selectButton.addListener(new Button.ClickListener() {
                public void buttonClick(ClickEvent event) {
                    Component viewComponent = null;
                    try {
                        if (EntityMenuItem.this.view instanceof Class) {
                            Class<Component> viewClass = (Class) EntityMenuItem.this.view;
                            viewComponent = viewClass.newInstance();
                            EntityMenuItem.this.view = viewComponent;
                        } else if (EntityMenuItem.this.view instanceof Component) {
                            viewComponent = (Component) EntityMenuItem.this.view;
                        }
                    } catch (InstantiationException e) {
                        Label l = new Label("Could not create view");
                        l.setSizeUndefined();
                        l.addStyleName(Reindeer.LABEL_H2);
                        viewComponent = l;
                    } catch (IllegalAccessException e) {
                        Label l = new Label("Could not create view");
                        l.setSizeUndefined();
                        l.addStyleName(Reindeer.LABEL_H2);
                        viewComponent = l;
                    }
                    setCurrentView(viewComponent);
                    for (EntityMenuItem item : menuItems) {
                        item.removeStyleName("open");
                        item.getNewButton().setEnabled(false);
                        item.getNewButton().setVisible(false);
                    }
                    EntityMenuItem.this.addStyleName("open");
                    if (EntityMenuItem.this.view instanceof AbstractEntityView) {
                        AbstractEntityView view = (AbstractEntityView) EntityMenuItem.this.view;
                        if (view.isCreateAllowed()) {
                            newButton.setEnabled(true);
                            newButton.setVisible(true);
                        }
                    }
                }
            });

            newButton.addListener(new Button.ClickListener() {
                public void buttonClick(ClickEvent event) {
                    // It's safe to assume the view is now a Component (the view
                    // has been opened at least once)
                    if (EntityMenuItem.this.view instanceof AbstractEntityView) {
                        AbstractEntityView view = (AbstractEntityView) EntityMenuItem.this.view;
                        view.createNewEntity();
                    }
                }
            });
        }

        public Button getNewButton() {
            return newButton;
        }
    }

    /**
     * First build the main layout and set the composition root to support the
     * visual editor. Then do any custom initialization.
     * 
     * The constructor will not be automatically regenerated by the visual
     * editor, and may be edited by the developer.
     */
    public WebEntityManagerView() {
        setSizeFull();

        // build the layout based on the visual editor
        buildMainLayout();
        setCompositionRoot(mainLayout);

        // add entity views to the list
        addEntityViewsToList();

        // TODO add user code here
    }

    /**
     * List all the entity views (classes annotated with
     * {@link RooVaadinEntityView}) and add them to the list on the left as
     * buttons. Clicking on one will open a new instance of that entity view in
     * the main view area.
     */
    private void addEntityViewsToList() {
        final Map<String, Class> entityViews = listEntityViews();
        if (entityViews.isEmpty()) {
            initialView.addComponent(new Label("No entity views exist"));
        } else {
            for (final String key : entityViews.keySet()) {
                // can either give a pre-created view or a view class
                Object view = entityViews.get(key);

                EntityMenuItem menuItem = new EntityMenuItem(key, view);
                menuItems.add(menuItem);
                viewList.addComponent(menuItem);
            }
            setCurrentView(null);
        }
    }

    /**
     * Sets the component to display as the main editor view.
     * 
     * @param view
     */
    protected void setCurrentView(Component view) {
        if (view == null) {
            view = initialView;
        }
        currentView = view;
        if (currentView instanceof ComponentContainer) {
            mainLayout.setSecondComponent(currentView);
        } else {
            VerticalLayout wrapper = new VerticalLayout();
            wrapper.setMargin(true);
            wrapper.setSizeFull();
            wrapper.addComponent(currentView);
            wrapper.setComponentAlignment(currentView, Alignment.MIDDLE_CENTER);
            mainLayout.setSecondComponent(wrapper);
        }
    }

    @AutoGenerated
    private HorizontalSplitPanel buildMainLayout() {
        // common part: create layout
        mainLayout = new HorizontalSplitPanel();
        mainLayout.addStyleName(Reindeer.SPLITPANEL_SMALL);
        mainLayout.addStyleName("main");

        // sidebar entity menu
        VerticalLayout sidebar = new VerticalLayout();
        sidebar.setSizeFull();
        sidebar.addStyleName(Reindeer.LAYOUT_BLUE);
        sidebar.addStyleName("sidebar");

        // viewList
        viewList = new CssLayout();
        viewList.setWidth("100%");
        viewList.addStyleName("view-list");

        Panel scroll = new Panel();
        scroll.addStyleName(Reindeer.PANEL_LIGHT);
        scroll.setSizeFull();
        scroll.setContent(viewList);

        // application header/name
        Label appName = new Label("Welcome", Label.CONTENT_XHTML);
        appName.addStyleName(Reindeer.LABEL_H1);
        viewList.addComponent(appName);

        sidebar.addComponent(appName);
        sidebar.addComponent(scroll);
        sidebar.setExpandRatio(scroll, 1);

        mainLayout.setFirstComponent(sidebar);

        // viewArea
        buildInitialView();
        mainLayout.setSecondComponent(initialView);
        mainLayout.setSplitPosition(17);

        return mainLayout;
    }

    @AutoGenerated
    private void buildInitialView() {
        // initialView
        initialView = new VerticalLayout();
        initialView.setMargin(true);
        initialView.setSizeFull();
        initialView.addStyleName(Reindeer.LAYOUT_BLUE);
        Label l = new Label(
                "<h1 class=\"v-label-h1\" style=\"text-align: center;\">Welcome</h1> Select an entity type from the left side menu to begin",
                Label.CONTENT_XHTML);
        l.setSizeUndefined();
        l.addStyleName(Reindeer.LABEL_SMALL);
        initialView.addComponent(l);
        initialView.setComponentAlignment(l, Alignment.MIDDLE_CENTER);
    }

}